/*
 * Decompiled with CFR 0.152.
 */
package jpos.basscom.utils;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import jpos.basscom.utils.BCrypt;

public class CryptoUtils {
    public static final String SECURE_RANDOM_ALGORITHM = "SHA1PRNG";
    public static final String KEY_BYTES_HASH_ALGORITHM = "SHA-256";
    public static final String AES_CIPHER = "AES/CBC/PKCS5Padding";
    public static final String AES_HMAC_ALGORITHM = "HmacSHA256";
    public static final String AES = "AES";
    public static final int AES_IV_LENGTH = 16;
    public static final int AES_DEFAULT_SIGNATURE_LENGTH = 16;
    public static final int AES_DEFAULT_ENCODING_KEY_LENGTH = 16;
    public static final int AES_DEFAULT_AUTH_KEY_LENGTH = 32;
    public static final int BCRYPT_DEFAULT_LOG_ROUNDS = 12;
    private static final char[] HEX_MAP = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
    private static SecureRandom random;
    private static AESKey cachedAESKey;

    public static byte[] generateAESKey() throws NoSuchAlgorithmException {
        return CryptoUtils.generateAESKey(16, 32);
    }

    public static byte[] generateAESKey(int encodingBytes, int authenticationBytes) throws NoSuchAlgorithmException {
        if (encodingBytes <= 0 || encodingBytes > 64) {
            throw new IllegalArgumentException("Encoding key length must be between 1 and 64 bytes");
        }
        if (authenticationBytes <= 0) {
            throw new IllegalArgumentException("Authentication key length must be greater than 0 bytes");
        }
        byte[] key = new byte[1 + encodingBytes + authenticationBytes];
        CryptoUtils.getRandom().nextBytes(key);
        key[0] = (byte)encodingBytes;
        return key;
    }

    public static byte[] generateAESKeyFromPassword(String password, String bcryptSalt) throws NoSuchAlgorithmException {
        return CryptoUtils.generateAESKeyFromPassword(password, bcryptSalt, 16, 32);
    }

    public static byte[] generateAESKeyFromPassword(String password, String bcryptSalt, int encodingBytes, int authenticationBytes) throws NoSuchAlgorithmException {
        int length;
        if (password == null || password.length() == 0 || bcryptSalt == null || bcryptSalt.length() == 0) {
            throw new IllegalArgumentException("Password and salt required");
        }
        if (encodingBytes <= 0 || encodingBytes > 64) {
            throw new IllegalArgumentException("Encoding key length must be between 1 and 64 bytes");
        }
        if (authenticationBytes <= 0) {
            throw new IllegalArgumentException("Authentication key length must be greater than 0 bytes");
        }
        byte[] key = new byte[1 + encodingBytes + authenticationBytes];
        key[0] = (byte)encodingBytes;
        byte[] digest = BCrypt.generateRawHash(password, bcryptSalt);
        for (int pos = 1; pos < key.length; pos += length) {
            byte[] oldDigest = digest;
            digest = MessageDigest.getInstance(KEY_BYTES_HASH_ALGORITHM).digest(oldDigest);
            Arrays.fill(oldDigest, (byte)0);
            length = Math.min(digest.length, key.length - pos);
            System.arraycopy(digest, 0, key, pos, length);
        }
        Arrays.fill(digest, (byte)0);
        return key;
    }

    public static String generateBCryptSalt() throws NoSuchAlgorithmException {
        return CryptoUtils.generateBCryptSalt(12);
    }

    public static String generateBCryptSalt(int logBase2rounds) throws NoSuchAlgorithmException {
        return BCrypt.gensalt(logBase2rounds, CryptoUtils.getRandom());
    }

    public static void setAESKey(byte[] key) {
        CryptoUtils.clearAESKey();
        cachedAESKey = new AESKey(key);
    }

    public static void clearAESKey() {
        if (cachedAESKey != null) {
            cachedAESKey.dispose();
            cachedAESKey = null;
        }
    }

    public static byte[] encryptAES(byte[] plainText) throws GeneralSecurityException {
        if (cachedAESKey == null) {
            throw new IllegalStateException("setAESKey(key) must be called before using encryptAES(data)");
        }
        return CryptoUtils.encryptAES(plainText, cachedAESKey);
    }

    public static byte[] encryptAES(byte[] plainText, byte[] key) throws GeneralSecurityException {
        AESKey aesKey = new AESKey(key);
        byte[] encryptedData = CryptoUtils.encryptAES(plainText, aesKey);
        aesKey.dispose();
        return encryptedData;
    }

    public static byte[] encryptAES(byte[] plainText, AESKey key) throws GeneralSecurityException {
        return CryptoUtils.encryptAES(plainText, key, 16);
    }

    public static byte[] encryptAES(byte[] plainText, AESKey key, int signatureLength) throws GeneralSecurityException {
        if (plainText == null || plainText.length == 0) {
            return new byte[0];
        }
        Cipher cipher = Cipher.getInstance(AES_CIPHER);
        byte[] iv = new byte[16];
        CryptoUtils.getRandom().nextBytes(iv);
        cipher.init(1, (Key)new SecretKeySpec(key.getEncodingKey(), AES), new IvParameterSpec(iv));
        byte[] cipherText = cipher.doFinal(plainText);
        byte[] signature = CryptoUtils.signCipherText(iv, cipherText, key.getAuthKey());
        return CryptoUtils.collateEncryptedData(iv, signature, cipherText, signatureLength);
    }

    public static byte[] decryptAES(byte[] encryptedData) throws GeneralSecurityException {
        if (cachedAESKey == null) {
            throw new IllegalStateException("setAESKey(key) must be called before using decryptAES(data)");
        }
        return CryptoUtils.decryptAES(encryptedData, cachedAESKey);
    }

    public static byte[] decryptAES(byte[] encryptedData, byte[] key) throws GeneralSecurityException {
        AESKey aesKey = new AESKey(key);
        byte[] plainText = CryptoUtils.decryptAES(encryptedData, aesKey);
        aesKey.dispose();
        return plainText;
    }

    public static byte[] decryptAES(byte[] encryptedData, AESKey key) throws GeneralSecurityException {
        return CryptoUtils.decryptAES(encryptedData, key, 16);
    }

    public static byte[] decryptAES(byte[] encryptedData, AESKey key, int expectedSignatureLength) throws GeneralSecurityException {
        if (encryptedData == null || encryptedData.length == 0) {
            return new byte[0];
        }
        Cipher cipher = Cipher.getInstance(AES_CIPHER);
        int pos = 0;
        byte ivLength = encryptedData[pos];
        if (ivLength != 16) {
            throw new GeneralSecurityException("IV length invalid");
        }
        byte[] iv = new byte[ivLength];
        System.arraycopy(encryptedData, ++pos, iv, 0, ivLength);
        byte signatureLength = encryptedData[pos += ivLength];
        if (signatureLength != expectedSignatureLength) {
            throw new GeneralSecurityException("Signature length invalid");
        }
        byte[] signature = new byte[signatureLength];
        System.arraycopy(encryptedData, ++pos, signature, 0, signatureLength);
        int cipherTextLength = encryptedData.length - (pos += signatureLength);
        byte[] cipherText = new byte[cipherTextLength];
        System.arraycopy(encryptedData, pos, cipherText, 0, cipherTextLength);
        if (!CryptoUtils.verifyCipherText(iv, cipherText, key.getAuthKey(), signature, signatureLength)) {
            throw new GeneralSecurityException("Cipher text signature invalid");
        }
        try {
            cipher.init(2, (Key)new SecretKeySpec(key.getEncodingKey(), AES), new IvParameterSpec(iv));
            return cipher.doFinal(cipherText);
        }
        catch (GeneralSecurityException e) {
            throw new GeneralSecurityException("Failed to decrypt");
        }
    }

    public static boolean areSignaturesEqual(byte[] signature1, byte[] signature2, int bytesToCompare) {
        if (signature1.length != signature2.length && (bytesToCompare <= 0 || signature1.length < bytesToCompare || signature2.length < bytesToCompare)) {
            return false;
        }
        int length = bytesToCompare > 0 && bytesToCompare < signature1.length ? bytesToCompare : signature1.length;
        boolean isEqual = true;
        for (int i = 0; i < length; ++i) {
            isEqual = signature1[i] == signature2[i] & isEqual;
        }
        return isEqual;
    }

    public static String convertToHex(byte[] bytes) {
        char[] hex = new char[bytes.length * 2];
        for (int i = 0; i < bytes.length; ++i) {
            char highHexDigit = HEX_MAP[(bytes[i] & 0xF0) >> 4];
            char lowHexDigit = HEX_MAP[bytes[i] & 0xF];
            hex[i * 2] = highHexDigit;
            hex[i * 2 + 1] = lowHexDigit;
        }
        return new String(hex);
    }

    public static byte[] convertFromHex(String hex) {
        if (hex.length() % 2 == 1) {
            throw new IllegalArgumentException("Hex string must have an even number of characters");
        }
        byte[] bytes = new byte[hex.length() / 2];
        for (int i = 0; i < hex.length(); i += 2) {
            int highBits = Character.digit(hex.charAt(i), 16);
            int lowBits = Character.digit(hex.charAt(i + 1), 16);
            bytes[i / 2] = (byte)((highBits << 4) + lowBits);
        }
        return bytes;
    }

    public static byte[] hmac256GenerateHash(byte[] authKey, byte[] value) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hmac = Mac.getInstance(AES_HMAC_ALGORITHM);
        hmac.init(new SecretKeySpec(authKey, AES_HMAC_ALGORITHM));
        hmac.update(value);
        return hmac.doFinal();
    }

    private static byte[] signCipherText(byte[] iv, byte[] cipherText, byte[] authKey) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hmac = Mac.getInstance(AES_HMAC_ALGORITHM);
        hmac.init(new SecretKeySpec(authKey, AES_HMAC_ALGORITHM));
        hmac.update(iv);
        hmac.update(cipherText);
        return hmac.doFinal();
    }

    private static boolean verifyCipherText(byte[] iv, byte[] cipherText, byte[] authKey, byte[] signature, int signatureLength) throws NoSuchAlgorithmException, InvalidKeyException {
        byte[] calcuatedSignature = CryptoUtils.signCipherText(iv, cipherText, authKey);
        return CryptoUtils.areSignaturesEqual(calcuatedSignature, signature, signatureLength);
    }

    private static byte[] collateEncryptedData(byte[] iv, byte[] signature, byte[] cipherText, int signatureLength) {
        byte[] encryptedData = new byte[18 + signatureLength + cipherText.length];
        int pos = 0;
        encryptedData[pos] = 16;
        System.arraycopy(iv, 0, encryptedData, ++pos, 16);
        encryptedData[pos += 16] = (byte)signatureLength;
        System.arraycopy(signature, 0, encryptedData, ++pos, signatureLength);
        System.arraycopy(cipherText, 0, encryptedData, pos += signatureLength, cipherText.length);
        Arrays.fill(iv, (byte)0);
        Arrays.fill(signature, (byte)0);
        Arrays.fill(cipherText, (byte)0);
        return encryptedData;
    }

    private static SecureRandom getRandom() throws NoSuchAlgorithmException {
        if (random == null) {
            random = SecureRandom.getInstance(SECURE_RANDOM_ALGORITHM);
        }
        return random;
    }

    public static class AESKey {
        private byte[] encodingKey;
        private byte[] authKey;

        public AESKey(byte[] keyBytes) {
            byte encodeLength = keyBytes[0];
            this.encodingKey = new byte[encodeLength];
            System.arraycopy(keyBytes, 1, this.encodingKey, 0, encodeLength);
            int authLength = keyBytes.length - encodeLength - 1;
            this.authKey = new byte[authLength];
            System.arraycopy(keyBytes, encodeLength + 1, this.authKey, 0, authLength);
            Arrays.fill(keyBytes, (byte)0);
        }

        public AESKey(byte[] encodingKey, byte[] authKey) {
            this.encodingKey = encodingKey;
            this.authKey = authKey;
        }

        public byte[] getEncodingKey() {
            return this.encodingKey;
        }

        public byte[] getAuthKey() {
            return this.authKey;
        }

        public void dispose() {
            Arrays.fill(this.encodingKey, (byte)0);
            Arrays.fill(this.authKey, (byte)0);
        }
    }
}

